//=============================================================================
// RPG Maker MZ - Cursor Application to Button Picture
//=============================================================================

/*:
 * @target MZ
 * @base ButtonPicture
 * @plugindesc Adds a cursor sptrite to a button picture when selected.
 * @author nz_prism
 *
 * @help CursorApplicationToButtonPicture.js
 * ver. 1.0.0
 *
 * This plugin provides a functionality to add a cursor sprite to
 * a button picture.
 * You can choose an image file (place an image in img/system) or an enclosing rectangle for a cursor.
 * This plugin requires ButtonPicture.js.
 * Place this plugin under ButtonPicture.js.
 *
 * Use it in the following procedure.
 *   1. Execute the procedure as described in ButonPicture.js.
 *   2. Configure the plugin parameters.
 *   2. When a mouse cursor enters a button picture, a cursor sprite appears.
 *
 * This plugin is released under the MIT License.
 * https://opensource.org/licenses/mit-license.php
 * 
 * 
 * @param useCursorImage
 * @text Use Cursor Image
 * @desc Set whether you use a cursor image or not.
 * @default false
 * @type boolean
 * 
 * @param cursorImageName
 * @text Cursor Image Name
 * @desc If you use a cursor image, choose a image file.
 * @default Cursor
 * @type file
 * @dir img/system/
 * 
 * @param isCursorXRight
 * @text Place Cursor Right
 * @desc If you use a cursor image, set whether you place the cursor at the right of the button picture. If false, it's placed at the left.
 * @default true
 * @type boolean
 * 
 * @param isCursorYDown
 * @text Place Cursor Down
 * @desc If you use a cursor image, set whether you place the cursor at the down of the button picture. If false, it's placed at the up.
 * @default true
 * @type boolean
 * 
 * @param borderWidth
 * @text Border Line Width
 * @desc If you don't use a cursor image, set the width of the border line.
 * @default 3
 * @type number
 * @min 0
 * 
 * @param borderColor
 * @text Border Line Color
 * @desc If you don't use a cursor image, set the color of the border line.
 * @default {"red":"255","green":"255","blue":"255","alpha":"0.5"}
 * @type struct<rgba>
 * 
 */

/*:ja
 * @target MZ
 * @plugindesc ピクチャボタンにマウスカーソルが重なった時カーソルを表示します。
 * @author nz_prism
 * @base ButtonPicture
 *
 * @help CursorApplicationToButtonPicture.js
 * ver. 1.0.0
 *
 * このプラグインは、ピクチャにマウスカーソルが重なった時にカーソルを
 * 表示する機能を提供します。
 * カーソルは画像（img/systemフォルダ）かピクチャを囲む枠線を選択できます。
 * このプラグインはButtonPicture.jsが導入されていることを前提とします。
 * ButtonPicture.jsよりも下に配置してください。
 *
 * 次の手順で使用してください。
 *   1. ButtonPicture.jsの手順通りに設定してください。
 *   2. プラグインパラメータを必要に応じて設定してください。
 *   3. ピクチャボタンにマウスカーソルが重なるとカーソルが表示されます。
 * 
 * このプラグインはMITライセンスにてリリースされています。
 * https://opensource.org/licenses/mit-license.php
 * 
 * 
 * @param useCursorImage
 * @text カーソル画像の使用
 * @desc カーソル画像を使用するかどうかを選択してください。
 * @default false
 * @type boolean
 * 
 * @param cursorImageName
 * @text カーソル画像
 * @desc カーソル画像を使用する場合、使用する画像を選択してください。画像はimg/systemフォルダに入れてください。
 * @default Cursor
 * @type file
 * @dir img/system/
 * 
 * @param isCursorXRight
 * @text カーソルX右寄せ
 * @desc カーソル画像を使用する場合、カーソル画像のX座標を画像本体の右端にするかどうかを設定してください。falseの場合、左端になります。
 * @default true
 * @type boolean
 * 
 * @param isCursorYDown
 * @text カーソルY下寄せ
 * @desc カーソル画像を使用する場合、カーソル画像のY座標を画像本体の下端にするかどうかを設定してください。falseの場合、上端になります。
 * @default true
 * @type boolean
 * 
 * @param borderWidth
 * @text 枠線太さ
 * @desc カーソル画像を使用しない場合、画像の外枠太さを指定してください。
 * @default 3
 * @type number
 * @min 0
 * 
 * @param borderColor
 * @text 枠線色
 * @desc カーソル画像を使用しない場合、画像外枠の色を指定してください。
 * @default {"red":"255","green":"255","blue":"255","alpha":"0.5"}
 * @type struct<rgba>
 * 
 */

/*~struct~rgba:
 *
 * @param red
 * @text Red
 * @desc The value of red.
 * @type number
 * @max 255
 * @min 0
 * 
 * @param green
 * @text Green
 * @desc The value of green.
 * @type number
 * @max 255
 * @min 0
 * 
 * @param blue
 * @text Blue
 * @desc The value of blue.
 * @type number
 * @max 255
 * @min 0
 * 
 * @param alpha
 * @text Alpha
 * @desc The value of alpha.
 * @type number
 * @max 255
 * @min 0
 * @decimals 1
 * 
 */

(() => {
    'use strict';

    const pluginName = "CursorApplicationToButtonPicture";


    const USE_CURSOR_IMAGE = PluginManager.parameters(pluginName).useCursorImage === "true";
    const CURSOR_IMAGE_NAME = PluginManager.parameters(pluginName).cursorImageName;
    const IS_CURSOR_X_RIGHT = PluginManager.parameters(pluginName).isCursorXRight === "true";
    const IS_CURSOR_Y_DOWN = PluginManager.parameters(pluginName).isCursorYDown === "true";
    const BORDER_WIDTH = Number(PluginManager.parameters(pluginName).borderWidth);
    const bordrColorObj = JSON.parse(PluginManager.parameters(pluginName).borderColor);
    const BORDER_COLOR = "rgba(" + bordrColorObj.red + "," + bordrColorObj.green + "," + bordrColorObj.blue + "," + bordrColorObj.alpha + ")";


    Sprite_Picture.prototype.createCursorSprite = function() {
        const mainBitmap = this.bitmap;
        if (mainBitmap) {
            const sprite = new Sprite();
            this._cursorSprite = sprite;
            this.addChild(sprite);
            const width = mainBitmap.width;
            const height = mainBitmap.height;
            let cursorBitmap;
            if (USE_CURSOR_IMAGE) {
                cursorBitmap = ImageManager.loadSystem(CURSOR_IMAGE_NAME);
                sprite.x = IS_CURSOR_X_RIGHT ? width : 0;
                sprite.y = IS_CURSOR_Y_DOWN ? height : 0;
            } else {
                cursorBitmap = new Bitmap(width + BORDER_WIDTH * 2, height + BORDER_WIDTH * 2);
                cursorBitmap.fillRect(0, 0, BORDER_WIDTH, height + BORDER_WIDTH, BORDER_COLOR);
                cursorBitmap.fillRect(0, 0, width + BORDER_WIDTH, BORDER_WIDTH, BORDER_COLOR);
                cursorBitmap.fillRect(width, 0, BORDER_WIDTH, height + BORDER_WIDTH, BORDER_COLOR);
                cursorBitmap.fillRect(0, height, width + BORDER_WIDTH, BORDER_WIDTH, BORDER_COLOR);
                sprite.x = -BORDER_WIDTH;
                sprite.y = -BORDER_WIDTH;
            }
            sprite.bitmap = cursorBitmap;
        }
    };

    const _Sprite_Picture_prototype_onMouseEnter = Sprite_Picture.prototype.onMouseEnter;
    Sprite_Picture.prototype.onMouseEnter = function() {
        _Sprite_Picture_prototype_onMouseEnter.call(this);
        if (!this._cursorSprite && this.picture().mzkp_commonEventId) {
            this.createCursorSprite();
        }
        this._cursorSprite.show();
    };

    const _Sprite_Picture_prototype_onMouseExit = Sprite_Picture.prototype.onMouseExit;
    Sprite_Picture.prototype.onMouseExit = function() {
        _Sprite_Picture_prototype_onMouseExit.call(this);
        if (this._cursorSprite) this._cursorSprite.hide();
    };

})();
